<?php

use Joomla\Filesystem\Exception\FilesystemException;
use Joomla\Filesystem\File;
use Joomla\Filesystem\Folder;
use Joomla\Filesystem\Path;

final class WatchfulliFileHelper
{
    public static function exists(string $path): bool
    {
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFile::exists($path);
        }

        return is_file(Path::clean($path));
    }

    public static function write(string $file, string $buffer): bool
    {
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFile::write($file, $buffer);
        }

        return File::write($file, $buffer);
    }

    public static function delete(string $file): bool
    {
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFile::delete($file);
        }

        if (self::exists($file) === false) {
            return true;
        }

        try {
            return File::delete($file);
        } catch (FilesystemException $e) {
            watchfulli::debug($e->getMessage());
            return false;
        }
    }

    public static function move(string $source, string $destination): bool
    {
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFile::move($source, $destination);
        }

        return File::move($source, $destination);
    }

    public static function getExt(string $file): string
    {
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFile::getExt($file);
        }

        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '5.0', '<')) {
            return \Joomla\CMS\Filesystem\File::getExt($file);
        }

        return File::getExt($file);
    }

    public static function folderExists(string $path): bool
    {
        return @is_dir($path);
    }

    # We do not use the J "Folder" library due to a bug when open_basedir is set GH 8415
    public static function copyFolderToRoot(string $src): bool
    {
        $dest = JPATH_ROOT;
        if (version_compare(WatchfulliJoomlaVersion::getVersion(), '3.10', '<')) {
            return JFolder::copy($src, $dest);
        }

        if (\function_exists('set_time_limit')) {
            set_time_limit(ini_get('max_execution_time'));
        }

        $src  = rtrim($src, \DIRECTORY_SEPARATOR);
        $dest = rtrim($dest, \DIRECTORY_SEPARATOR);

        if (!is_dir(Path::clean($src))) {
            throw new FilesystemException('Source folder not found', -1);
        }

        if (!($dh = @opendir($src))) {
            throw new FilesystemException('Cannot open source folder', -1);
        }

        // Walk through the directory copying files and recursing into folders.
        while (($file = readdir($dh)) !== false) {
            $sfid = $src . '/' . $file;
            $dfid = $dest . '/' . $file;

            switch (filetype($sfid)) {
                case 'dir':
                    if ($file != '.' && $file != '..') {
                        $ret = Folder::copy($sfid, $dfid,  '', true);

                        if ($ret !== true) {
                            return $ret;
                        }
                    }

                    break;

                case 'file':
                        if (!@copy($sfid, $dfid)) {
                            throw new FilesystemException('Copy file failed', -1);
                        }

                    if (version_compare(WatchfulliJoomlaVersion::getMajorVersion(), '4.0', '>')) {
                        File::invalidateFileCache($dfid);
                    }
                    break;
            }
        }

        return true;
    }

	public static function isWindows(): bool
	{
		return strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';
	}
}